<?php
namespace Services;

class TwitterClient
{
    private string $bearerToken;

    public function __construct(string $bearerToken)
    {
        $this->bearerToken = $bearerToken;
    }

    public function fetchUserTweets(string $username, ?string $sinceId = null, int $limit = 5, bool $includeRetweets = true, bool $includeReplies = true): array
    {
        if (empty($this->bearerToken)) {
            throw new \Exception('Twitter Bearer Token is empty.');
        }

        $queryParts = ['from:' . $username];
        if (!$includeRetweets) {
            $queryParts[] = '-is:retweet';
        }
        if (!$includeReplies) {
            $queryParts[] = '-is:reply';
        }
        $queryStr = implode(' ', $queryParts);

        $url = "https://api.x.com/2/tweets/search/recent";
        $query = [
            'query' => $queryStr,
            'max_results' => $limit,
            'tweet.fields' => 'id,text,created_at,attachments',
            'expansions' => 'attachments.media_keys,author_id',
            'media.fields' => 'url,preview_image_url'
        ];

        if ($sinceId) {
            $query['since_id'] = $sinceId;
        }

        $url .= '?' . http_build_query($query);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->bearerToken,
            ]
        ]);

        $response = curl_exec($ch);
        if ($response === false) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new \Exception('Twitter API error: ' . $error);
        }
        curl_close($ch);

        $data = json_decode($response, true);
        if (!$data) {
            return [];
        }
        return $data;
    }
}
