<?php
namespace App\Http;

/**
 * HttpClient (No-Composer)
 *
 * هدف الملف: تقليل/حل 403 من متاجر سلة عبر:
 * - User-Agent متصفح حقيقي
 * - Headers شبيهة بالمتصفح
 * - Cookie jar (حفظ/إرسال الكوكيز)
 * - Referer + Warm-up request (زيارة الصفحة الرئيسية أولاً)
 */
class HttpClient
{
    private string $cookieFile;

    public function __construct(private string $userAgent)
    {
        // لو الـ UA كان عام/بوت، بدّله بـ UA متصفح حقيقي
        if (stripos($this->userAgent, 'SallaMigrator') !== false || stripos($this->userAgent, 'compatible;') !== false) {
            $this->userAgent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0 Safari/537.36';
        }
        $tmpDir = dirname(__DIR__, 2) . '/storage/tmp';
        if (!is_dir($tmpDir)) {
            @mkdir($tmpDir, 0775, true);
        }
        $this->cookieFile = rtrim($tmpDir, '/') . '/cookies_' . substr(md5($this->userAgent . microtime(true)), 0, 12) . '.txt';
    }

    public function get(string $url): string
    {
        // أول محاولة
        $res = $this->curlRequest($url);
        if ($res['ok']) return $res['body'];

        // لو 403 أو 429/302 أحيانًا، جرّب warm-up ثم إعادة المحاولة
        if (in_array($res['code'], [403, 429], true)) {
            $base = $this->baseFromUrl($url);
            if ($base) {
                // Warm-up: افتح الهوم لتحصيل كوكيز/جلسة
                $this->curlRequest($base, $base);
                // إعادة المحاولة مع referer
                $res2 = $this->curlRequest($url, $base);
                if ($res2['ok']) return $res2['body'];
                throw new \RuntimeException("Fetch failed ({$res2['code']}): {$res2['error']}");
            }
        }

        throw new \RuntimeException("Fetch failed ({$res['code']}): {$res['error']}");
    }

    public function download(string $url, string $destPath): void
    {
        $dir = dirname($destPath);
        if (!is_dir($dir)) @mkdir($dir, 0775, true);

        $fp = fopen($destPath, 'wb');
        if (!$fp) throw new \RuntimeException('Cannot open file for writing: ' . $destPath);

        $base = $this->baseFromUrl($url);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_FILE => $fp,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_CONNECTTIMEOUT => 20,
            CURLOPT_TIMEOUT => 90,
            CURLOPT_USERAGENT => $this->userAgent,
            CURLOPT_ENCODING => '',
            CURLOPT_COOKIEJAR => $this->cookieFile,
            CURLOPT_COOKIEFILE => $this->cookieFile,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_HTTPHEADER => [
                'Accept: image/avif,image/webp,image/apng,image/*,*/*;q=0.8',
                'Accept-Language: ar,en-US;q=0.9,en;q=0.8',
                'Cache-Control: no-cache',
                'Pragma: no-cache',
                'Connection: keep-alive',
            ],
        ]);
        if ($base) {
            curl_setopt($ch, CURLOPT_REFERER, $base);
        }

        $ok = curl_exec($ch);
        $err = curl_error($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        fclose($fp);

        if ($ok === false || $code >= 400) {
            @unlink($destPath);
            throw new \RuntimeException("Download failed ($code): $err");
        }
    }

    /**
     * Internal cURL request.
     * @return array{ok:bool,code:int,body:string,error:string}
     */
    private function curlRequest(string $url, ?string $referer = null): array
    {
        $ch = curl_init($url);

        $headers = [
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
            'Accept-Language: ar,en-US;q=0.9,en;q=0.8',
            'Cache-Control: no-cache',
            'Pragma: no-cache',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
        ];

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_CONNECTTIMEOUT => 20,
            CURLOPT_TIMEOUT => 40,
            CURLOPT_USERAGENT => $this->userAgent,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_ENCODING => '',
            CURLOPT_COOKIEJAR => $this->cookieFile,
            CURLOPT_COOKIEFILE => $this->cookieFile,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);
        if ($referer) {
            curl_setopt($ch, CURLOPT_REFERER, $referer);
        }

        $body = curl_exec($ch);
        $err = curl_error($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $ok = ($body !== false) && ($code >= 200 && $code < 300);

        return [
            'ok' => $ok,
            'code' => $code,
            'body' => $body === false ? '' : (string)$body,
            'error' => $err,
        ];
    }

    private function baseFromUrl(string $url): string
    {
        $p = parse_url($url);
        if (!$p || empty($p['host'])) return '';
        $scheme = $p['scheme'] ?? 'https';
        return $scheme . '://' . $p['host'];
    }
}
