<?php
namespace App\Storage;

use PDO;
use Throwable;

class Db
{
    private PDO $pdo;

    public function __construct(array $config)
    {
        $dsn = sprintf(
            'mysql:host=%s;dbname=%s;charset=%s',
            $config['host'],
            $config['name'],
            $config['charset'] ?? 'utf8mb4'
        );

        $this->pdo = new PDO($dsn, $config['user'], $config['pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
    }

    public function pdo(): PDO
    {
        return $this->pdo;
    }

    /**
     * Insert or update a product (dedupe by source_url and/or sku)
     * Returns product id.
     */
    public function upsertProduct(array $p): int
    {
        $sql = "INSERT INTO products (source, source_url, source_product_id, sku, name, description_html, description_text, price, currency, brand, availability, categories_json, main_image_url, status, last_scraped_at)
                VALUES (:source, :source_url, :source_product_id, :sku, :name, :description_html, :description_text, :price, :currency, :brand, :availability, :categories_json, :main_image_url, :status, NOW())
                ON DUPLICATE KEY UPDATE
                  source_product_id = COALESCE(VALUES(source_product_id), source_product_id),
                  sku = COALESCE(VALUES(sku), sku),
                  name = VALUES(name),
                  description_html = COALESCE(VALUES(description_html), description_html),
                  description_text = COALESCE(VALUES(description_text), description_text),
                  price = COALESCE(VALUES(price), price),
                  currency = VALUES(currency),
                  brand = COALESCE(VALUES(brand), brand),
                  availability = COALESCE(VALUES(availability), availability),
                  categories_json = COALESCE(VALUES(categories_json), categories_json),
                  main_image_url = COALESCE(VALUES(main_image_url), main_image_url),
                  status = IF(status='exported','exported',VALUES(status)),
                  last_scraped_at = NOW();";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            ':source' => $p['source'] ?? 'salla',
            ':source_url' => $p['source_url'],
            ':source_product_id' => $p['source_product_id'] ?? null,
            ':sku' => $p['sku'] ?? null,
            ':name' => $p['name'] ?? '',
            ':description_html' => $p['description_html'] ?? null,
            ':description_text' => $p['description_text'] ?? null,
            ':price' => $p['price'] ?? null,
            ':currency' => $p['currency'] ?? 'SAR',
            ':brand' => $p['brand'] ?? null,
            ':availability' => $p['availability'] ?? null,
            ':categories_json' => isset($p['categories']) ? json_encode($p['categories'], JSON_UNESCAPED_UNICODE) : null,
            ':main_image_url' => $p['main_image_url'] ?? null,
            ':status' => $p['status'] ?? 'new',
        ]);

        $id = (int)$this->pdo->lastInsertId();
        if ($id > 0) return $id;

        // For duplicate update, retrieve id by source_url
        $q = $this->pdo->prepare('SELECT id FROM products WHERE source_url = ? LIMIT 1');
        $q->execute([$p['source_url']]);
        $row = $q->fetch();
        return (int)($row['id'] ?? 0);
    }

    public function upsertImage(int $productId, string $url, ?string $localPath, int $order): void
    {
        $sql = "INSERT INTO product_images (product_id, image_url, local_path, sort_order)
                VALUES (:pid, :url, :path, :ord)
                ON DUPLICATE KEY UPDATE
                  local_path = COALESCE(VALUES(local_path), local_path),
                  sort_order = VALUES(sort_order)";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            ':pid' => $productId,
            ':url' => $url,
            ':path' => $localPath,
            ':ord' => $order,
        ]);
    }

    public function getProducts(array $filters = []): array
    {
        $where = [];
        $params = [];
        if (!empty($filters['q'])) {
            $where[] = '(name LIKE :q OR sku LIKE :q OR source_url LIKE :q)';
            $params[':q'] = '%' . $filters['q'] . '%';
        }
        if (!empty($filters['status'])) {
            $where[] = 'status = :status';
            $params[':status'] = $filters['status'];
        }
        $sql = 'SELECT * FROM products';
        if ($where) $sql .= ' WHERE ' . implode(' AND ', $where);
        $sql .= ' ORDER BY updated_at DESC LIMIT 1000';

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    public function getProduct(int $id): ?array
    {
        $stmt = $this->pdo->prepare('SELECT * FROM products WHERE id = ?');
        $stmt->execute([$id]);
        $p = $stmt->fetch();
        if (!$p) return null;
        $i = $this->pdo->prepare('SELECT * FROM product_images WHERE product_id = ? ORDER BY sort_order ASC');
        $i->execute([$id]);
        $p['images'] = $i->fetchAll();
        $p['categories'] = $p['categories_json'] ? json_decode($p['categories_json'], true) : [];
        return $p;
    }

    public function stats(): array
    {
        $out = [];
        $out['total'] = (int)$this->pdo->query('SELECT COUNT(*) c FROM products')->fetch()['c'];
        $out['new'] = (int)$this->pdo->query("SELECT COUNT(*) c FROM products WHERE status='new'")->fetch()['c'];
        $out['exported'] = (int)$this->pdo->query("SELECT COUNT(*) c FROM products WHERE status='exported'")->fetch()['c'];
        $out['failed'] = (int)$this->pdo->query("SELECT COUNT(*) c FROM products WHERE status='failed'")->fetch()['c'];
        return $out;
    }

    public function markExported(array $ids): void
    {
        if (!$ids) return;
        $in = implode(',', array_fill(0, count($ids), '?'));
        $stmt = $this->pdo->prepare("UPDATE products SET status='exported' WHERE id IN ($in)");
        $stmt->execute(array_values($ids));
    }
}
