<?php
namespace App\Utils;

use App\Http\HttpClient;
use App\Parser\CategoryParser;
use App\Parser\ProductParser;
use App\Storage\Db;
use App\Storage\ImageDownloader;

class Scraper
{
    public function __construct(
        private HttpClient $http,
        private Db $db,
        private Logger $logger,
        private ImageDownloader $img
    ) {}

    public function scrapeCategory(string $url): int
    {
        $html = $this->http->get($url);
        $parser = new CategoryParser();
        // Base URL from category URL
        $base = $this->baseFromUrl($url);
        $items = $parser->parse($html, $base);

        $added = 0;
        foreach ($items as $it) {
            // Save minimal then queue product scrape by calling directly
            try {
                $pid = $this->db->upsertProduct([
                    'source' => 'salla',
                    'source_url' => $it['source_url'],
                    'source_product_id' => $it['source_product_id'] ?? null,
                    'name' => $it['name'] ?? '',
                    'price' => $it['price'] ?? null,
                    'currency' => 'SAR',
                    'main_image_url' => $it['main_image_url'] ?? null,
                    'status' => 'new'
                ]);
                if ($pid) $added++;
            } catch (\Throwable $e) {
                $this->logger->error('scrapeCategory', $e->getMessage());
            }
        }
        $this->logger->info('scrapeCategory', "Parsed {$added} products from category.");
        return $added;
    }

    public function scrapeProduct(string $url, bool $downloadImages = true): int
    {
        $html = $this->http->get($url);
        $parser = new ProductParser();
        $p = $parser->parse($html);

        if (empty($p['source_url'])) {
            $p['source_url'] = $url;
        }

        $productId = $this->db->upsertProduct([
            'source' => 'salla',
            'source_url' => $p['source_url'],
            'sku' => $p['sku'] ?? null,
            'name' => $p['name'] ?? '',
            'description_html' => $p['description_html'] ?? null,
            'description_text' => $p['description_text'] ?? null,
            'price' => $p['price'] ?? null,
            'currency' => $p['currency'] ?? 'SAR',
            'brand' => $p['brand'] ?? null,
            'availability' => $p['availability'] ?? null,
            'categories' => $p['categories'] ?? [],
            'main_image_url' => $p['main_image_url'] ?? null,
            'status' => 'new'
        ]);

        $images = $p['images'] ?? [];
        $order = 1;
        if ($images) {
            $locals = $downloadImages ? $this->img->downloadProductImages($productId, $images) : [];
            $localMap = [];
            foreach ($locals as $l) {
                $localMap[$l['url']] = $l['path'];
            }
            foreach ($images as $imgUrl) {
                $this->db->upsertImage($productId, $imgUrl, $localMap[$imgUrl] ?? null, $order++);
            }
        }

        $this->logger->info('scrapeProduct', "Scraped product #{$productId}");
        return $productId;
    }

    
/**
 * Parse product HTML string (from browser capture) and save to DB.
 */
public function scrapeProductHtml(string $html, string $fallbackUrl = '', bool $downloadImages = true): int
{
    $parser = new ProductParser();
    $p = $parser->parse($html);

    if (empty($p['name'])) {
        throw new \RuntimeException('Not a product page (missing product name).');
    }

    if (empty($p['source_url'])) {
        $p['source_url'] = $fallbackUrl ?: '';
    }

    $productId = $this->db->upsertProduct([
        'source' => 'salla',
        'source_url' => $p['source_url'],
        'sku' => $p['sku'] ?? null,
        'name' => $p['name'] ?? '',
        'description_html' => $p['description_html'] ?? null,
        'description_text' => $p['description_text'] ?? null,
        'price' => $p['price'] ?? null,
        'currency' => $p['currency'] ?? 'SAR',
        'availability' => $p['availability'] ?? null,
        'brand' => $p['brand'] ?? null,
        'categories_json' => json_encode($p['categories'] ?? [], JSON_UNESCAPED_UNICODE),
        'main_image_url' => $p['images'][0] ?? ($p['main_image_url'] ?? null),
        'status' => 'new'
    ]);

    if ($downloadImages && !empty($p['images']) && $productId) {
        $this->img->downloadAll($productId, $p['images'], $p['source_url'] ?? $fallbackUrl);
    }

    $this->logger->info('scrapeProductHtml', "Saved product #{$productId} from browser HTML");
    return $productId;
}

/**
 * Parse category HTML string (from browser capture) and save minimal products to DB.
 */
public function scrapeCategoryHtml(string $html, string $categoryUrl): int
{
    $parser = new CategoryParser();
    $base = $this->baseFromUrl($categoryUrl);
    $items = $parser->parse($html, $base);

    $added = 0;
    foreach ($items as $it) {
        try {
            $pid = $this->db->upsertProduct([
                'source' => 'salla',
                'source_url' => $it['source_url'],
                'source_product_id' => $it['source_product_id'] ?? null,
                'name' => $it['name'] ?? '',
                'price' => $it['price'] ?? null,
                'currency' => 'SAR',
                'main_image_url' => $it['main_image_url'] ?? null,
                'status' => 'new'
            ]);
            if ($pid) $added++;
        } catch (\Throwable $e) {
            $this->logger->error('scrapeCategoryHtml', $e->getMessage());
        }
    }
    $this->logger->info('scrapeCategoryHtml', "Parsed {$added} products from browser category HTML.");
    return $added;
}

public function scrapeProductHtmlFile(string $filePath, bool $downloadImages = false): int
    {
        $html = file_get_contents($filePath);
        if ($html === false) throw new \RuntimeException('Cannot read file: ' . $filePath);
        $parser = new ProductParser();
        $p = $parser->parse($html);
        $productId = $this->db->upsertProduct([
            'source' => 'html',
            'source_url' => $p['source_url'] ?? ('file://' . basename($filePath)),
            'sku' => $p['sku'] ?? null,
            'name' => $p['name'] ?? basename($filePath),
            'description_html' => $p['description_html'] ?? null,
            'description_text' => $p['description_text'] ?? null,
            'price' => $p['price'] ?? null,
            'currency' => $p['currency'] ?? 'SAR',
            'brand' => $p['brand'] ?? null,
            'availability' => $p['availability'] ?? null,
            'categories' => $p['categories'] ?? [],
            'main_image_url' => $p['main_image_url'] ?? null,
            'status' => 'new'
        ]);

        $images = $p['images'] ?? [];
        $order = 1;
        if ($images) {
            $locals = $downloadImages ? $this->img->downloadProductImages($productId, $images) : [];
            $localMap = [];
            foreach ($locals as $l) $localMap[$l['url']] = $l['path'];
            foreach ($images as $imgUrl) {
                $this->db->upsertImage($productId, $imgUrl, $localMap[$imgUrl] ?? null, $order++);
            }
        }

        return $productId;
    }

    private function baseFromUrl(string $url): string
    {
        $p = parse_url($url);
        if (!$p) return '';
        $scheme = $p['scheme'] ?? 'https';
        $host = $p['host'] ?? '';
        return $scheme . '://' . $host;
    }
}
