<?php
namespace App\Parser;

use Symfony\Component\DomCrawler\Crawler;

class CategoryParser
{
    /**
     * Extract product cards from category/listing html.
     * Returns array of items: [source_product_id, url, name, price, image]
     */
    public function parse(string $html, string $baseUrl = ''): array
    {
        $c = new Crawler($html);
        $items = [];

        // Each product card is a custom element: custom-salla-product-card
        $c->filter('custom-salla-product-card')->each(function(Crawler $card) use (&$items, $baseUrl) {
            $sourceId = $card->attr('id') ?: null;
            $url = null;
            if ($card->filter('a')->count()) {
                $url = $card->filter('a')->first()->attr('href');
                if ($url && $baseUrl && str_starts_with($url, '/')) {
                    $url = rtrim($baseUrl,'/') . $url;
                }
            }

            $name = null;
            if ($card->filter('h3.s-product-card-content-title a')->count()) {
                $name = trim($card->filter('h3.s-product-card-content-title a')->text());
            }

            $price = null;
            if ($card->filter('h4.s-product-card-price')->count()) {
                $raw = trim($card->filter('h4.s-product-card-price')->text());
                // extract number
                if (preg_match('/([0-9]+(\.[0-9]+)?)/', str_replace(',', '', $raw), $m)) {
                    $price = $m[1];
                }
            }

            $image = null;
            // prefer data-src (cdn)
            if ($card->filter('img')->count()) {
                $img = $card->filter('img')->first();
                $image = $img->attr('data-src') ?: $img->attr('src');
            }

            if ($url) {
                $items[] = [
                    'source_product_id' => $sourceId,
                    'source_url' => $url,
                    'name' => $name,
                    'price' => $price,
                    'main_image_url' => (is_string($image) && preg_match('~^https?://~', $image)) ? $image : null,
                ];
            }
        });

        return $items;
    }
}
