<?php
namespace Services;

use Core\DB;
use Services\TwitterClient;
use Services\TelegramClient;

class SyncService
{
    public function __construct()
    {
    }

    public function syncAll(): void
    {
        $sourcesStmt = DB::query("
            SELECT ts.*, sa.api_key AS bearer_token, sa.name AS account_name
            FROM twitter_sources ts
            INNER JOIN social_accounts sa ON sa.id = ts.social_account_id
            WHERE ts.status = 'active'
        ");
        $sources = $sourcesStmt->fetchAll();

        foreach ($sources as $source) {
            $bearer = $source['bearer_token'] ?? '';
            if (empty($bearer)) {
                continue;
            }
            $twitter = new TwitterClient($bearer);
            $this->syncSource($source, $twitter);
        }
    }

    private function syncSource(array $source, TwitterClient $twitter): void
    {
        $username = $source['identifier'];
        $sinceId = $source['last_tweet_id'] ?? null;
        $includeRetweets = (int)$source['include_retweets'] === 1;
        $includeReplies = (int)$source['include_replies'] === 1;

        $data = $twitter->fetchUserTweets($username, $sinceId, 20, $includeRetweets, $includeReplies);
        if (empty($data['data'])) {
            return;
        }

        $tweets = array_reverse($data['data']);
        $media = $data['includes']['media'] ?? [];

        $mediaIndex = [];
        foreach ($media as $m) {
            if (isset($m['media_key'])) {
                $mediaIndex[$m['media_key']] = $m;
            }
        }

        $maxId = $sinceId;

        $rulesStmt = DB::query("
            SELECT pr.*, tt.chat_id, sa.bot_token, tt.display_name AS target_name
            FROM publishing_rules pr
            INNER JOIN telegram_targets tt ON tt.id = pr.telegram_target_id
            INNER JOIN social_accounts sa ON sa.id = tt.social_account_id
            WHERE pr.twitter_source_id = :sid AND pr.enabled = 1
        ", ['sid' => $source['id']]);
        $rules = $rulesStmt->fetchAll();

        foreach ($tweets as $tweet) {
            $tweetId = $tweet['id'];
            $text = $tweet['text'];

            $tweetUrl = 'https://x.com/' . $username . '/status/' . $tweetId;

            $images = [];
            if (!empty($tweet['attachments']['media_keys'])) {
                foreach ($tweet['attachments']['media_keys'] as $key) {
                    if (isset($mediaIndex[$key]['url'])) {
                        $images[] = $mediaIndex[$key]['url'];
                    }
                }
            }

            $existsStmt = DB::query('SELECT id FROM posts WHERE twitter_source_id = :sid AND tweet_id = :tid LIMIT 1', [
                'sid' => $source['id'],
                'tid' => $tweetId
            ]);
            $postId = null;
            if ($row = $existsStmt->fetch()) {
                $postId = $row['id'];
            } else {
                DB::query('INSERT INTO posts (twitter_source_id, tweet_id, tweet_text, tweet_url, media_json, created_at, updated_at) 
                    VALUES (:sid, :tid, :txt, :url, :media, NOW(), NOW())', [
                    'sid' => $source['id'],
                    'tid' => $tweetId,
                    'txt' => $text,
                    'url' => $tweetUrl,
                    'media' => json_encode($images, JSON_UNESCAPED_UNICODE),
                ]);
                $postId = DB::conn()->lastInsertId();
            }

            foreach ($rules as $rule) {
                $telegram = new TelegramClient($rule['bot_token']);

                $finalText = $rule['text_template']
                    ? str_replace(['{text}', '{url}'], [$text, $tweetUrl], $rule['text_template'])
                    : $text . "\n\n🔗 " . $tweetUrl;

                try {
                    if (!empty($images) && (int)$rule['post_with_media'] === 1) {
                        $res = $telegram->sendPhoto($rule['chat_id'], $images[0], $finalText);
                    } else {
                        $res = $telegram->sendMessage($rule['chat_id'], $finalText);
                    }

                    DB::query('INSERT INTO post_logs (post_id, publishing_rule_id, telegram_message_id, status, sent_text, target_chat_id, created_at) 
                        VALUES (:pid, :rid, :mid, :st, :txt, :tchat, NOW())', [
                        'pid' => $postId,
                        'rid' => $rule['id'],
                        'mid' => $res['result']['message_id'] ?? null,
                        'st' => 'success',
                        'txt' => $finalText,
                        'tchat' => $rule['chat_id']
                    ]);
                } catch (\Throwable $e) {
                    DB::query('INSERT INTO post_logs (post_id, publishing_rule_id, status, error_message, sent_text, target_chat_id, created_at) 
                        VALUES (:pid, :rid, :st, :err, :txt, :tchat, NOW())', [
                        'pid' => $postId,
                        'rid' => $rule['id'],
                        'st' => 'failed',
                        'err' => $e->getMessage(),
                        'txt' => $finalText,
                        'tchat' => $rule['chat_id']
                    ]);
                }
            }

            if (!$maxId || $tweetId > $maxId) {
                $maxId = $tweetId;
            }
        }

        if ($maxId && $maxId !== $sinceId) {
            DB::query('UPDATE twitter_sources SET last_tweet_id = :mid, updated_at = NOW() WHERE id = :id', [
                'mid' => $maxId,
                'id' => $source['id']
            ]);
        }
    }
}
